<?php
class PhpForm {
	protected static $nextId = 0;
	
	public static function getNextId () {
		return 'id-' . ++self::$nextId;
	}
	
	protected	$template_dir	= null,
				$json_file		= null,
				$object			= null;
	
	/**
	 * initialise a form
	 * $template_dir	: directory containing widgets templates
	 * $json_file		: description of the form
	 * $object			: the object injected in form fields
	 */
	public function __construct ( string|null $template_dir = null, string|null $json_file = null, $object = null ) {
		$this->setTemplateDir( $template_dir );
		$this->setJsonFile( $json_file );
		$this->setObject( $object );
	}
	
	public function setTemplateDir ( $template_dir ) {
		$this->template_dir = $template_dir;
	}
	
	public function getTemplateDir () {
		return $this->template_dir;
	}
	
	public function setJsonFile ( $json_file ) {
		$this->json_file = $json_file;
	}
	
	public function getJsonFile () {
		return $this->json_file;
	}
	
	public function setObject ( $object ) {
		$this->object = $object;
	}
	
	public function getObject () {
		return $this->object;
	}
	
	public function __toString () {
		// tempalte directory
		$template_dir = $this->getTemplateDir();
		if ( !is_dir( $template_dir ) ) {
			error_log( 'Template directory "' . $template_dir . '" is not a directory' );
			return '';
		}
		
		// JSon File
		$json_file = $this->getJsonFile();
		if ( !file_exists( $json_file ) ) {
			error_log( 'Json File "' . $json_file . '" is not a file.' );
			return '';
		}
		$content = file_get_contents( $json_file );
		if ( empty( $content ) ) {
			error_log( 'Json File "' . $json_file . '" is empty or not readable.' );
			return '';
		}
		$form = json_decode( $content, true );
		if ( json_last_error() !== 0 ) {
			error_log( 'Json File "' . $json_file . '" contain errors.' );
			return '';
		}
		
		$object = $this->getObject();
		if ( !is_null( $object ) && !is_a( $object, $form[ 'object' ] ) ) {
			error_log( 'The object is not a "' . $form[ 'object' ] . '"' );
			return '';
		}
		
		return $this->createForm( $form[ 'content' ] );
	}
	
	public function createForm ( $form ) {
		$html = '';
		$object = $this->getObject();
		if ( is_null( $object ) ) {
			$object = new StdClass();
		}
		
		foreach ( $form as $widget ) {
			if ( !isset( $widget[ 'type' ] ) ) {
				error_log( 'A widget has no type :' . PHP_EOL . print_r( $widget, true ) );
				continue;
			}
			$html .= $this->printWidget( $widget );
		}
		
		return $html;
	}
	
	protected function printWidget ( $widget ) {
		$file = $this->getTemplateDir() . DIRECTORY_SEPARATOR . $widget[ 'type' ] . '.php';
		if ( !file_exists( $file ) ) {
			error_log( 'Template "' . $file . '" does not exist' );
			return '';
		}
		
		extract( $widget );
		ob_start();
		include( $file );
		return ob_get_clean();
	}
	
	protected function getObjectValue ( $attr, $default_value = '' ) {
		$object = $this->getObject();
		if ( is_null( $object ) ) {
			return $default_value;
		}
		
		if ( property_exists( $object, $attr ) || method_exists( $object, '__get' ) ) {
			return $object->$attr;
		}
		return $default_value;
	}
	
	/* check */
	public function check ( $post, $files ) {
		// JSon File
		$json_file = $this->getJsonFile();
		if ( !file_exists( $json_file ) ) {
			error_log( 'Json File "' . $json_file . '" is not a file.' );
			return '';
		}
		$content = file_get_contents( $json_file );
		if ( empty( $content ) ) {
			error_log( 'Json File "' . $json_file . '" is empty or not readable.' );
			return '';
		}
		$form = json_decode( $content, true );
		if ( json_last_error() !== 0 ) {
			error_log( 'Json File "' . $json_file . '" contain errors.' );
			return '';
		}
		
		$errors = [];
		
		// mandatory fields :
		foreach ( $this->extractMandatoryFields( $form ) as $row ) {
			if ( empty( $post[ $row[ 'name' ] ] ) ) {
				$errors[] = 'Le champ «<a href="#id-' . $row[ 'name' ] . '">' . $row[ 'label' ] . '</a>» est obligatoire.';
			}
		}
		
		// manual checks :
		foreach ( $this->extractArraysWithKey( $form, 'check' ) as $row ) {
			$check = $row[ 'check' ];
			if ( isset( $check[ 'regex' ] ) || isset( $check[ 'range' ] ) ) {
				$check = [ $check ];
			}
			
			if ( isset( $row[ 'mandatory' ] ) && $row[ 'mandatory' ] && empty( $post[ $row[ 'name' ] ] ) ) {
				continue;
			}
			
			foreach ( $check as $verif ) {
				if ( isset( $verif[ 'regex' ] ) ) {
					if ( !preg_match( '#' . $verif[ 'regex' ] . '#', $post[ $row[ 'name' ] ] ) ) {
						$errors[] = 'Le champ «<a href="#id-' . $row[ 'name' ] . '">' . $row[ 'label' ] . '</a>» contient une erreur.';
					}
				}
			}
		}
		
		// dates format :
		foreach ( $this->extractArraysWithType( $form, 'date' ) as $row ) {
			if ( ( !isset( $row[ 'mandatory' ] ) || !$row[ 'mandatory' ] ) && empty( $post[ $row[ 'name' ] ] ) ) {
				continue;
			}
			
			if ( !preg_match( '#\d{2}/\d{2}/\d{4}#', $post[ $row[ 'name' ] ] ) ) {
				if ( isset( $row[ 'mandatory' ] ) && $row[ 'mandatory' ] && empty( $post[ $row[ 'name' ] ] ) ) {
					continue;
				}
				
				$errors[] = 'La date «<a href="#id-' . $row[ 'name' ] . '">' . $row[ 'label' ] . '</a>» est mal formatée.';
			}
		}
		
		foreach ( $this->extractArraysWithType( $form, 'double_date' ) as $row ) {
			if ( ( !isset( $row[ 'mandatory' ] ) || !$row[ 'mandatory' ] ) && empty( $post[ $row[ 'name1' ] ] ) ) {
				continue;
			}
			
			if ( !preg_match( '#\d{2}/\d{2}/\d{4}#', $post[ $row[ 'name1' ] ] ) ) {
				if ( ( !isset( $row[ 'mandatory' ] ) || !$row[ 'mandatory' ] ) && !empty( $post[ $row[ 'name1' ] ] ) ) {
					$errors[] = 'Pour le champ «<a href="#id-' . $row[ 'name' ] . '">' . $row[ 'label' ] . '</a>», la date de début est mal formatée.';
				}
			}

			if ( !preg_match( '#\d{2}/\d{2}/\d{4}#', $post[ $row[ 'name2' ] ] ) ) {
				if ( ( !isset( $row[ 'mandatory' ] ) || !$row[ 'mandatory' ] ) && !empty( $post[ $row[ 'name2' ] ] ) ) {
					$errors[] = 'Pour le champ «<a href="#id-' . $row[ 'name' ] . '">' . $row[ 'label' ] . '</a>», la date de fin est mal formatée.';
				}
			}
		}
		
		// select and radio values :
		$fields = array_merge( $this->extractArraysWithType( $form, 'select' ), $this->extractArraysWithType( $form, 'selectpicker' ), $this->extractArraysWithType( $form, 'radio' ) );
		foreach ( $fields as $row ) {
			if ( ( !isset( $row[ 'mandatory' ] ) || !$row[ 'mandatory' ] ) && !empty( $post[ $row[ 'name' ] ] ) ) {
				continue;
			}
			
			$values = [];
			if ( isset( $row[ 'values' ] ) ) {
				$values = $row[ 'values' ];
			} else if ( isset( $row[ 'dico' ] ) ) {
				$dico = getDicoByName( $row[ 'dico' ] );
				$values = array_combine( array_column( $dico, 'dico_value' ), array_column( $dico, 'dico_label' ) );
			} else if ( isset( $row[ 'callback_values' ] ) ) {
				$values = $row[ 'callback_values' ]();
			}
			
			if ( !in_array( $post[ $row[ 'name' ] ], array_keys( $values ) ) ) {
				$errors[] = 'Pour le champ «<a href="#id-' . $row[ 'name' ] . '">' . $row[ 'label' ] . '</a>», la valeur n\'est pas autorisée.';
			}
		}
		
		// element check :
		if ( method_exists( $form[ 'object' ], 'check' ) ) {
			$reflectionMethod = new ReflectionMethod( $form[ 'object' ], 'check' );
			if ( $reflectionMethod->isStatic() ) {
				$check = $form[ 'object' ]::check( $post, $files );
			} else {
				$object = new $form[ 'object' ]( $post, $files );
				$check = $object->check( $post, $files );
			}
			
			if ( $check !== true ) {
				$errors = array_merge( $errors, $check );
			}
		}
		// /element check :
		
		if ( count( $errors ) > 0 ) {
			return $errors;
		}
		return true;
	}
	
	protected function extractMandatoryFields ( $array ) {
		$result = [];
		
		foreach ( $array as $item ) {
			if ( is_array( $item ) ) {
				if ( isset( $item[ 'mandatory' ] ) && $item[ 'mandatory' ] ) {
					$result[] = $item;
				}
				$result = array_merge( $result, $this->extractMandatoryFields( $item ) );
			}
		}
		
		return $result;
	}
	
	protected function extractArraysWithKey ( $array, $key ) {
		$result = [];
		
		foreach ( $array as $item ) {
			if ( is_array( $item ) ) {
				if ( array_key_exists( $key, $item ) ) {
					$result[] = $item;
				}
				$result = array_merge( $result, $this->extractArraysWithKey( $item, $key ) );
			}
		}
		
		return $result;
	}

	protected function extractArraysWithType ( $array, $type ) {
		$result = [];
		
		foreach ( $array as $item ) {
			if ( is_array( $item ) ) {
				if ( isset( $item[ 'type' ] ) && ( $item[ 'type' ] === $type ) ) {
					$result[] = $item;
				}
				$result = array_merge( $result, $this->extractArraysWithType( $item, $type ) );
			}
		}
		
		return $result;
	}
	/* /check */
	
	/* save */
	public function save ( $post, $files ) {
		$json_file = $this->getJsonFile();
		$content = file_get_contents( $json_file );
		$form = json_decode( $content, true );
		
		foreach ( $this->extractArraysWithType( $form, 'date' ) as $row ) {
			$post[ $row[ 'name' ] ] = formatDate( $post[ $row[ 'name' ] ], 'd/m/Y', 'Y-m-d' );
		}
		foreach ( $this->extractArraysWithType( $form, 'double_date' ) as $row ) {
			$post[ $row[ 'name1' ] ] = formatDate( $post[ $row[ 'name1' ] ], 'd/m/Y', 'Y-m-d' );
			$post[ $row[ 'name2' ] ] = formatDate( $post[ $row[ 'name2' ] ], 'd/m/Y', 'Y-m-d' );
		}
		
		// element save :
		$errors = [];
		if ( method_exists( $form[ 'object' ], 'save' ) ) {
			$reflectionMethod = new ReflectionMethod( $form[ 'object' ], 'save' );
			if ( $reflectionMethod->isStatic() ) {
				$save = $form[ 'object' ]::save( $post, $files );
			} else {
				$object = new $form[ 'object' ]( $post, $files );
				$save = $object->save( $post, $files );
			}
			
			if ( $save !== true ) {
				$errors = array_merge( $errors, $save );
			}
		} else {
			error_log( 'La methode '. $form[ 'object' ] . '::save n\'existe pas.' );
		}
		// /element save :
		
		if ( count( $errors ) > 0 ) {
			return $errors;
		}
		return true;
	}
	/* /save */
}
