/* Dark mode automatique */
function invertColors ( e ) {
	var computedStyle = window.getComputedStyle( e );
	if ( computedStyle.backgroundColor !== 'rgba(0, 0, 0, 0)' ) {
		e.dataset[ 'o_bg_color' ] = computedStyle.backgroundColor;
		e.dataset[ 'n_bg_color' ] = invertColorPerceptual( computedStyle.backgroundColor );
	}
	if ( computedStyle.borderTopColor !== 'rgba(0, 0, 0, 0)' ) {
		e.dataset[ 'o_bt_color' ] = computedStyle.borderTopColor;
		e.dataset[ 'n_bt_color' ] = invertColorPerceptual( computedStyle.borderTopColor );
	}
	if ( computedStyle.borderLeftColor !== 'rgba(0, 0, 0, 0)' ) {
		e.dataset[ 'o_bl_color' ] = computedStyle.borderLeftColor;
		e.dataset[ 'n_bl_color' ] = invertColorPerceptual( computedStyle.borderLeftColor );
	}
	if ( computedStyle.borderBottomColor !== 'rgba(0, 0, 0, 0)' ) {
		e.dataset[ 'o_bb_color' ] = computedStyle.borderBottomColor;
		e.dataset[ 'n_bb_color' ] = invertColorPerceptual( computedStyle.borderBottomColor );
	}
	if ( computedStyle.borderRightColor !== 'rgba(0, 0, 0, 0)' ) {
		e.dataset[ 'o_br_color' ] = computedStyle.borderRightColor;
		e.dataset[ 'n_br_color' ] = invertColorPerceptual( computedStyle.borderRightColor );
	}
	
	Array.from( e.children ).forEach( invertColors );
}

function rgbToHsl ( r, g, b ) {
	r /= 255; g /= 255; b /= 255;
	var max = Math.max( r, g, b ),
		min = Math.min( r, g, b ),
		h, s, l = ( max + min ) / 2;
	
	if ( max === min ) {
		h = s = 0; // achromatic
	} else {
		var d = max - min;
		s = l > 0.5 ? d / ( 2 - max - min ) : d / ( max + min );
		switch ( max ) {
			case r : h = ( g - b ) / d + ( g < b ? 6 : 0 ); break;
			case g : h = ( b - r ) / d + 2; break;
			case b : h = ( r - g ) / d + 4; break;
		}
		h /= 6;
	}
	return [ h, s, l ];
}

function hslToRgb ( h, s, l ) {
	var r, g, b;
	if ( s === 0 ) {
		r = g = b = l; // achromatic
	} else {
		function hue2rgb( p, q, t ) {
			if ( t < 0 ) t += 1;
			if ( t > 1 ) t -= 1;
			if ( t < 1/6 ) return p + ( q - p ) * 6 * t;
			if ( t < 1/2 ) return q;
			if ( t < 2/3 ) return p + ( q - p ) * ( 2/3 - t ) * 6;
			return p;
		}
		var q = l < 0.5 ? l * ( 1 + s ) : l + s - l * s,
			p = 2 * l - q;
		r = hue2rgb( p, q, h + 1/3 );
		g = hue2rgb( p, q, h );
		b = hue2rgb( p, q, h - 1/3 );
	}
	return [
		Math.round( r * 255 ),
		Math.round( g * 255 ),
		Math.round( b * 255 )
	];
}

function getEffectiveBackgroundColor ( e ) {
	while ( e && ( e !== document ) ) {
		var bg = e.dataset[ 'n_bg_color' ];
		// Si la couleur n'est pas totalement transparente
		if ( bg && !/^rgba?\(0,\s*0,\s*0,\s*0\)$/.test(bg) && ( bg !== 'transparent' ) ) {
			return bg;
		}
		e = e.parentElement;
	}
	return 'rgb(255,255,255)'; // Fond par défaut (blanc)
}

function luminance ( r, g, b ) {
	// Formule standard WCAG pour la luminance relative
	var a = [r, g, b].map( function(v) {
		v /= 255;
		return v <= 0.03928 ? v / 12.92 : Math.pow( ( v + 0.055 ) / 1.055, 2.4 );
	} );
	return 0.2126 * a[ 0 ] + 0.7152 * a[ 1 ] + 0.0722 * a[ 2 ];
}

function getContrast ( rgb1, rgb2 ) {
	// rgb1/rgb2 : tableau [r,g,b]
	var	lum1 = luminance( rgb1[ 0 ], rgb1[ 1 ], rgb1[ 2 ] ) + 0.05,
		lum2 = luminance( rgb2[ 0 ], rgb2[ 1 ], rgb2[ 2 ] ) + 0.05;
	return lum1 > lum2 ? lum1 / lum2 : lum2 / lum1;
}

function parseRgb ( str ) {
	// Accepte 'rgb(r,g,b)' ou 'rgba(r,g,b,a)'
	var m = str.match(/\d+/g);
	if ( !m ) {
		return [ 255, 255, 255 ];
	}
	return [ parseInt( m[ 0 ] ), parseInt( m[ 1 ] ), parseInt( m[ 2 ] ) ];
}

function getTextColorForBackground ( bgColor ) {
	var rgb = parseRgb( bgColor ),
		// Inverse la luminance pour le dark mode
		[ h, s, l ] = rgbToHsl( rgb[ 0 ], rgb[ 1 ], rgb[ 2 ] ),
		invertedL = 1 - l,
		[ ri, gi, bi ] = hslToRgb( h, s, invertedL ),
		// Vérifie le contraste avec le fond inversé
		contrast = getContrast( [ ri, gi, bi ], rgb );
	
	// Si le contraste est insuffisant (<4.5 recommandé par WCAG), force noir ou blanc
	if ( contrast < 4.5 ) {
		var whiteContrast = getContrast( [ 255, 255, 255 ], rgb ) ,
			blackContrast = getContrast( [ 0, 0, 0 ], rgb );
		return whiteContrast > blackContrast ? 'rgb(255,255,255)' : 'rgb(0,0,0)';
	}
	return `rgb(${ri}, ${gi}, ${bi})`;
}

function invertColorPerceptual ( rgb ) {
	// Accepte rgb(r, g, b) ou rgba(r, g, b, a)
	var match = rgb.match( /\d+(\.\d+)?/g );
	if ( !match ) return rgb;
	var [ r, g, b ] = match.map( Number ),
		a = match[ 3 ] !== undefined && match.length > 3 ? Number( match[ 3 ] ) : undefined;
	
	// Conversion perceptuelle
	var [ h, s, l ] = rgbToHsl( r, g, b );
	
	// Inversion de la luminance
	l = 1 - l;
	var [ ri, gi, bi ] = hslToRgb( h, s, l );
	
	if ( a !== undefined ) {
		return `rgba(${ri}, ${gi}, ${bi}, ${a})`;
	}
	
	return `rgb(${ri}, ${gi}, ${bi})`;
}

function applyDarkModeWithContrast ( e ) {
	var bg = getEffectiveBackgroundColor( e );
	if ( e.nodeType === 1 ) {
		var style = window.getComputedStyle( e );
		if ( style.color ) {
			e.dataset[ 'o_color' ] = style.color;
			e.dataset[ 'n_color' ] = getTextColorForBackground( bg );
		}
		
		Array.from( e.children ).forEach( applyDarkModeWithContrast );
	}
}

function switchMode () {
	if ( document.body.classList.contains( 'dark-mode-on' ) ) {
		document.querySelectorAll( '[data-n_bg_color]' ).forEach( function ( e ) {
			e.style.backgroundColor = e.dataset[ 'n_bg_color' ];
		} );
		document.querySelectorAll( '[data-n_bt_color]' ).forEach( function ( e ) {
			e.style.borderTopColor = e.dataset[ 'n_bt_color' ];
		} );
		document.querySelectorAll( '[data-n_bl_color]' ).forEach( function ( e ) {
			e.style.borderLeftColor = e.dataset[ 'n_bl_color' ];
		} );
		document.querySelectorAll( '[data-n_bb_color]' ).forEach( function ( e ) {
			e.style.borderBottomColor = e.dataset[ 'n_bb_color' ];
		} );
		document.querySelectorAll( '[data-n_br_color]' ).forEach( function ( e ) {
			e.style.borderRightColorColor = e.dataset[ 'n_br_color' ];
		} );
		document.querySelectorAll( '[data-n_color]' ).forEach( function ( e ) {
			e.style.color = e.dataset[ 'n_color' ];
		} );
		document.querySelectorAll('img').forEach( function ( img ) {
			img.style.filter = 'invert(1) hue-rotate(180deg)';
		} );
	} else {
		document.querySelectorAll( '[data-o_bg_color]' ).forEach( function ( e ) {
			e.style.backgroundColor = e.dataset[ 'o_bg_color' ];
		} );
		document.querySelectorAll( '[data-o_bt_color]' ).forEach( function ( e ) {
			e.style.borderTopColor = e.dataset[ 'o_bt_color' ];
		} );
		document.querySelectorAll( '[data-o_bl_color]' ).forEach( function ( e ) {
			e.style.borderLeftColor = e.dataset[ 'o_bl_color' ];
		} );
		document.querySelectorAll( '[data-o_bb_color]' ).forEach( function ( e ) {
			e.style.borderBottomColor = e.dataset[ 'o_bb_color' ];
		} );
		document.querySelectorAll( '[data-o_br_color]' ).forEach( function ( e ) {
			e.style.borderRightColorColor = e.dataset[ 'o_br_color' ];
		} );
		document.querySelectorAll( '[data-o_color]' ).forEach( function ( e ) {
			e.style.color = e.dataset[ 'o_color' ];
		} );
		document.querySelectorAll('img').forEach( function ( img ) {
			img.style.filter = 'none';
		} );
	}
}

function convertToDarkMode ( target ) {
	if ( !document.body.classList.contains( 'dark-mode-calculated' ) ) {
		invertColors( target );
		applyDarkModeWithContrast( target );
		document.body.classList.add( 'dark-mode-calculated' );
	}
}

function initDarkMode () {
	var observer = new MutationObserver( function ( mutations ) {
		var elements = [];
		mutations.forEach( function ( e ) {
			elements.push( e.target );
		} );
		
		elements = elements.filter( function ( item, index, self ) {
			return index === self.findIndex( ( el ) => el.id === item.id )
		} );
		
		elements.forEach( function ( e ) {
			invertColors( e );
			applyDarkModeWithContrast( e );
		} );
		
		switchMode();
	} );

	// observer.observe( document.body, { childList: true, subtree: true } );
	convertToDarkMode( document.body );
}

function toogleMode () {
	if ( document.body.classList.contains( 'dark-mode-on' ) ) {
		document.body.classList.remove( 'dark-mode-on' );
	} else {
		document.body.classList.add( 'dark-mode-on' );
	}
	
	switchMode();
}


window.addEventListener( 'load', function () {
	initDarkMode();
	
	if ( document.querySelectorAll( 'html' )[ 0 ].dataset[ 'theme' ] == 'dark' ) {
		toogleMode();
		document.querySelectorAll( '.switch-theme-light' ).forEach( function ( n ) {
			n.style.display = 'none';
		} );
	} else {
		document.querySelectorAll( '.switch-theme-dark' ).forEach( function ( n ) {
			n.style.display = 'none';
		} );
	}
	
	document.querySelectorAll( '.switch-theme-light' ).forEach( function ( n ) {
		n.addEventListener( 'click', function () {
			document.querySelectorAll( 'html' )[ 0 ].dataset[ 'theme' ] = 'light';
			toogleMode();
			document.querySelectorAll( '.switch-theme-dark' ).forEach( function ( n ) {
				n.style.display = 'inline';
			} );
			document.querySelectorAll( '.switch-theme-light' ).forEach( function ( n ) {
				n.style.display = 'none';
			} );
		} );
	} );
	
	document.querySelectorAll( '.switch-theme-dark' ).forEach( function ( n ) {
		n.addEventListener( 'click', function () {
			document.querySelectorAll( 'html' )[ 0 ].dataset[ 'theme' ] = 'dark';
			toogleMode();
			document.querySelectorAll( '.switch-theme-light' ).forEach( function ( n ) {
				n.style.display = 'inline';
			} );
			document.querySelectorAll( '.switch-theme-dark' ).forEach( function ( n ) {
				n.style.display = 'none';
			} );
		} );
	} );
} );
